<?php
require_once __DIR__ . '/../app/bootstrap.php';

$ROUTES = [];

// --- Routes (Public) ---
route('GET', '/', function() use ($db, $config) {
  $cats = $db->query("SELECT * FROM categories ORDER BY sort_order ASC")->fetchAll();
  $new = $db->query("SELECT p.*, c.name category_name FROM products p
      LEFT JOIN categories c ON c.id=p.category_id
      WHERE p.is_active=1 ORDER BY p.updated_at DESC LIMIT 8")->fetchAll();
  include __DIR__ . '/../app/views/public/home.php';
});

route('GET', '/c/{slug}', function($params) use ($db) {
  $slug = $params['slug'];
  $cat = $db->prepare("SELECT * FROM categories WHERE slug=?");
  $cat->execute([$slug]);
  $cat = $cat->fetch();
  if (!$cat) { http_response_code(404); echo "Categoría no encontrada"; exit; }

  $sizes = $db->prepare("SELECT DISTINCT size FROM variants v
    JOIN products p ON p.id=v.product_id WHERE p.is_active=1 AND p.category_id=? AND size IS NOT NULL AND size<>'' ORDER BY size");
  $sizes->execute([$cat['id']]);
  $sizes = array_values(array_filter(array_map(fn($r)=>$r['size'], $sizes->fetchAll())));

  $colors = $db->prepare("SELECT DISTINCT color FROM variants v
    JOIN products p ON p.id=v.product_id WHERE p.is_active=1 AND p.category_id=? AND color IS NOT NULL AND color<>'' ORDER BY color");
  $colors->execute([$cat['id']]);
  $colors = array_values(array_filter(array_map(fn($r)=>$r['color'], $colors->fetchAll())));

  $q = trim($_GET['q'] ?? '');
  $f_size = $_GET['size'] ?? '';
  $f_color = $_GET['color'] ?? '';
  $sort = $_GET['sort'] ?? 'relevance';
  $where = "p.is_active=1 AND p.category_id=:cat_id";
  $join = "";
  $bind = [':cat_id' => $cat['id']];

  if ($q !== '') {
    $where .= " AND (p.name LIKE :q OR p.description LIKE :q)";
    $bind[':q'] = '%' . $q . '%';
  }

  if ($f_size !== '' || $f_color !== '') {
    $join = "JOIN variants v ON v.product_id=p.id";
    if ($f_size !== '') { $where .= " AND v.size=:size"; $bind[':size'] = $f_size; }
    if ($f_color !== '') { $where .= " AND v.color=:color"; $bind[':color'] = $f_color; }
  }

  $order = "p.updated_at DESC";
  if ($sort === 'price_asc') $order = "p.price ASC";
  if ($sort === 'price_desc') $order = "p.price DESC";

  $stmt = $db->prepare("SELECT DISTINCT p.* FROM products p $join WHERE $where ORDER BY $order LIMIT 48");
  $stmt->execute($bind);
  $products = $stmt->fetchAll();

  include __DIR__ . '/../app/views/public/category.php';
});

route('GET', '/p/{slug}', function($params) use ($db) {
  $slug = $params['slug'];
  $stmt = $db->prepare("SELECT p.*, c.name category_name, c.slug category_slug
    FROM products p LEFT JOIN categories c ON c.id=p.category_id
    WHERE p.slug=? AND p.is_active=1");
  $stmt->execute([$slug]);
  $p = $stmt->fetch();
  if (!$p) { http_response_code(404); echo "Producto no encontrado"; exit; }

  $imgs = $db->prepare("SELECT * FROM product_images WHERE product_id=? ORDER BY sort_order ASC, id ASC");
  $imgs->execute([$p['id']]);
  $imgs = $imgs->fetchAll();

  $vars = $db->prepare("SELECT * FROM variants WHERE product_id=? ORDER BY id ASC");
  $vars->execute([$p['id']]);
  $vars = $vars->fetchAll();

  include __DIR__ . '/../app/views/public/product.php';
});

route('GET', '/search', function() use ($db) {
  $q = trim($_GET['q'] ?? '');
  $products = [];
  if ($q !== '') {
    $stmt = $db->prepare("SELECT p.* FROM products p WHERE p.is_active=1 AND (p.name LIKE :q OR p.description LIKE :q) ORDER BY p.updated_at DESC LIMIT 48");
    $stmt->execute([':q'=>'%'.$q.'%']);
    $products = $stmt->fetchAll();
  }
  include __DIR__ . '/../app/views/public/search.php';
});

route('GET', '/cart', function() use ($db, $config) {
  $cart = $_SESSION['cart'] ?? [];
  $items = [];
  $subtotal = 0.0;

  foreach ($cart as $key => $row) {
    $pid = (int)($row['product_id'] ?? 0);
    $vid = (int)($row['variant_id'] ?? 0);
    $qty = max(1, (int)($row['qty'] ?? 1));

    $p = $db->prepare("SELECT * FROM products WHERE id=?");
    $p->execute([$pid]);
    $p = $p->fetch();
    if (!$p) continue;

    $v = null;
    if ($vid) {
      $st = $db->prepare("SELECT * FROM variants WHERE id=? AND product_id=?");
      $st->execute([$vid, $pid]);
      $v = $st->fetch();
    }

    $unit = (float)($v['price'] ?? $p['price']);
    $line = $unit * $qty;
    $subtotal += $line;

    $img = $db->prepare("SELECT url FROM product_images WHERE product_id=? ORDER BY sort_order ASC LIMIT 1");
    $img->execute([$pid]);
    $img = $img->fetchColumn();

    $items[] = [
      'key'=>$key,
      'product'=>$p,
      'variant'=>$v,
      'qty'=>$qty,
      'unit'=>$unit,
      'line'=>$line,
      'img'=>$img,
    ];
  }

  include __DIR__ . '/../app/views/public/cart.php';
});

route('POST', '/cart/add', function() use ($config) {
  require_post();
  if (!csrf_check($config)) { http_response_code(403); echo "CSRF inválido"; exit; }

  $pid = (int)($_POST['product_id'] ?? 0);
  $vid = (int)($_POST['variant_id'] ?? 0);
  $qty = max(1, (int)($_POST['qty'] ?? 1));

  $key = $pid . ':' . $vid;
  if (empty($_SESSION['cart'][$key])) {
    $_SESSION['cart'][$key] = ['product_id'=>$pid, 'variant_id'=>$vid, 'qty'=>$qty];
  } else {
    $_SESSION['cart'][$key]['qty'] += $qty;
  }
  redirect('/cart');
});

route('POST', '/cart/update', function() use ($config) {
  require_post();
  if (!csrf_check($config)) { http_response_code(403); echo "CSRF inválido"; exit; }

  $qtys = $_POST['qty'] ?? [];
  foreach ($qtys as $key => $q) {
    $q = (int)$q;
    if ($q <= 0) unset($_SESSION['cart'][$key]);
    else $_SESSION['cart'][$key]['qty'] = $q;
  }
  redirect('/cart');
});

route('GET', '/checkout', function() use ($config) {
  include __DIR__ . '/../app/views/public/checkout.php';
});

route('POST', '/checkout', function() use ($db, $config) {
  require_post();
  if (!csrf_check($config)) { http_response_code(403); echo "CSRF inválido"; exit; }

  $name = trim($_POST['name'] ?? '');
  $email = trim($_POST['email'] ?? '');
  $phone = trim($_POST['phone'] ?? '');
  $address = trim($_POST['address'] ?? '');
  $notes = trim($_POST['notes'] ?? '');

  $cart = $_SESSION['cart'] ?? [];
  if (!$cart) { flash_set('error', 'Tu carrito está vacío'); redirect('/cart'); }

  $items = [];
  $subtotal = 0.0;
  foreach ($cart as $key => $row) {
    $pid = (int)$row['product_id']; $vid = (int)$row['variant_id']; $qty = max(1, (int)$row['qty']);
    $p = $db->prepare("SELECT * FROM products WHERE id=?");
    $p->execute([$pid]); $p = $p->fetch();
    if (!$p) continue;
    $v = null;
    if ($vid) { $st = $db->prepare("SELECT * FROM variants WHERE id=?"); $st->execute([$vid]); $v = $st->fetch(); }
    $unit = (float)($v['price'] ?? $p['price']);
    $line = $unit * $qty;
    $subtotal += $line;
    $items[] = [$pid, $vid ?: null, $p['name'] . ($v ? " ({$v['color']} {$v['size']})" : ''), $qty, $unit, $line];
  }
  $total = $subtotal;

  $code = 'ORD-' . strtoupper(bin2hex(random_bytes(4)));
  $stmt = $db->prepare("INSERT INTO orders(code, status, customer_name, customer_email, customer_phone, shipping_address, notes, subtotal, total, created_at)
    VALUES(?,?,?,?,?,?,?,?,?,?)");
  $stmt->execute([$code, 'new', $name, $email, $phone, $address, $notes, $subtotal, $total, gmdate('c')]);
  $order_id = (int)$db->lastInsertId();

  $ins = $db->prepare("INSERT INTO order_items(order_id, product_id, variant_id, name, qty, unit_price, line_total)
    VALUES(?,?,?,?,?,?,?)");
  foreach ($items as $it) {
    $ins->execute([$order_id, $it[0], $it[1], $it[2], $it[3], $it[4], $it[5]]);
  }

  unset($_SESSION['cart']);
  include __DIR__ . '/../app/views/public/checkout_success.php';
});

// --- Admin routes ---
route('GET', '/admin', function() { redirect('/admin/dashboard'); });

route('GET', '/admin/login', function() use ($config) {
  include __DIR__ . '/../app/views/admin/login.php';
});

route('POST', '/admin/login', function() use ($db, $config) {
  require_post();
  if (!csrf_check($config)) { http_response_code(403); echo "CSRF inválido"; exit; }
  $email = trim($_POST['email'] ?? '');
  $pass = $_POST['password'] ?? '';
  if (admin_login($db, $email, $pass)) redirect('/admin/dashboard');
  flash_set('error', 'Credenciales inválidas');
  redirect('/admin/login');
});

route('POST', '/admin/logout', function() use ($config) {
  require_post();
  if (!csrf_check($config)) { http_response_code(403); echo "CSRF inválido"; exit; }
  admin_logout();
  redirect('/admin/login');
});

route('GET', '/admin/dashboard', function() use ($db) {
  admin_require();
  $today = gmdate('Y-m-d');
  $sales = $db->prepare("SELECT COALESCE(SUM(total),0) s FROM orders WHERE substr(created_at,1,10)=?");
  $sales->execute([$today]);
  $sales = (float)$sales->fetchColumn();

  $orders = (int)$db->query("SELECT COUNT(*) FROM orders")->fetchColumn();
  $low = (int)$db->query("SELECT COUNT(*) FROM variants WHERE stock <= 3")->fetchColumn();
  $products = (int)$db->query("SELECT COUNT(*) FROM products")->fetchColumn();

  include __DIR__ . '/../app/views/admin/dashboard.php';
});

route('GET', '/admin/products', function() use ($db) {
  admin_require();
  $items = $db->query("SELECT p.*, c.name category_name FROM products p LEFT JOIN categories c ON c.id=p.category_id ORDER BY p.updated_at DESC")->fetchAll();
  include __DIR__ . '/../app/views/admin/products.php';
});

route('GET', '/admin/products/new', function() use ($db) {
  admin_require();
  $cats = $db->query("SELECT * FROM categories ORDER BY sort_order ASC")->fetchAll();
  $p = ['id'=>null,'name'=>'','slug'=>'','description'=>'','price'=>0,'category_id'=>null,'is_active'=>1];
  $imgs = [];
  $vars = [];
  include __DIR__ . '/../app/views/admin/product_edit.php';
});

route('GET', '/admin/products/edit/{id}', function($params) use ($db) {
  admin_require();
  $id = (int)$params['id'];
  $cats = $db->query("SELECT * FROM categories ORDER BY sort_order ASC")->fetchAll();
  $st = $db->prepare("SELECT * FROM products WHERE id=?");
  $st->execute([$id]);
  $p = $st->fetch();
  if (!$p) { http_response_code(404); echo "Producto no encontrado"; exit; }

  $imgs = $db->prepare("SELECT * FROM product_images WHERE product_id=? ORDER BY sort_order ASC,id ASC");
  $imgs->execute([$id]); $imgs = $imgs->fetchAll();
  $vars = $db->prepare("SELECT * FROM variants WHERE product_id=? ORDER BY id ASC");
  $vars->execute([$id]); $vars = $vars->fetchAll();
  include __DIR__ . '/../app/views/admin/product_edit.php';
});

route('POST', '/admin/products/save', function() use ($db, $config) {
  admin_require();
  require_post();
  if (!csrf_check($config)) { http_response_code(403); echo "CSRF inválido"; exit; }

  $id = (int)($_POST['id'] ?? 0);
  $name = trim($_POST['name'] ?? '');
  $slug = trim($_POST['slug'] ?? '');
  $desc = trim($_POST['description'] ?? '');
  $price = (float)($_POST['price'] ?? 0);
  $cat = ($_POST['category_id'] ?? '') !== '' ? (int)$_POST['category_id'] : null;
  $active = isset($_POST['is_active']) ? 1 : 0;

  if ($slug === '') $slug = slugify($name);
  $now = gmdate('c');

  if ($id > 0) {
    $st = $db->prepare("UPDATE products SET name=?, slug=?, description=?, price=?, category_id=?, is_active=?, updated_at=? WHERE id=?");
    $st->execute([$name, $slug, $desc, $price, $cat, $active, $now, $id]);
  } else {
    $st = $db->prepare("INSERT INTO products(name, slug, description, price, category_id, is_active, created_at, updated_at) VALUES(?,?,?,?,?,?,?,?)");
    $st->execute([$name, $slug, $desc, $price, $cat, $active, $now, $now]);
    $id = (int)$db->lastInsertId();
  }

  $db->prepare("DELETE FROM product_images WHERE product_id=?")->execute([$id]);
  $img_urls = $_POST['img_url'] ?? [];
  $ins = $db->prepare("INSERT INTO product_images(product_id, url, sort_order) VALUES(?,?,?)");
  foreach ($img_urls as $i => $u) {
    $u = trim($u);
    if ($u === '') continue;
    $ins->execute([$id, $u, (int)$i]);
  }

  $db->prepare("DELETE FROM variants WHERE product_id=?")->execute([$id]);
  $var_sku = $_POST['var_sku'] ?? [];
  $var_color = $_POST['var_color'] ?? [];
  $var_size = $_POST['var_size'] ?? [];
  $var_price = $_POST['var_price'] ?? [];
  $var_stock = $_POST['var_stock'] ?? [];
  $insv = $db->prepare("INSERT INTO variants(product_id, sku, color, size, price, stock) VALUES(?,?,?,?,?,?)");
  $n = max(count($var_sku), count($var_color), count($var_size));
  for ($i=0; $i<$n; $i++) {
    $sku = trim($var_sku[$i] ?? '');
    $color = trim($var_color[$i] ?? '');
    $size = trim($var_size[$i] ?? '');
    $vprice = trim($var_price[$i] ?? '') === '' ? null : (float)$var_price[$i];
    $stock = (int)($var_stock[$i] ?? 0);
    if ($sku==='' && $color==='' && $size==='') continue;
    $insv->execute([$id, $sku, $color, $size, $vprice, $stock]);
  }

  flash_set('ok', 'Producto guardado');
  redirect('/admin/products/edit/' . $id);
});

route('POST', '/admin/products/delete', function() use ($db, $config) {
  admin_require();
  require_post();
  if (!csrf_check($config)) { http_response_code(403); echo "CSRF inválido"; exit; }
  $id = (int)($_POST['id'] ?? 0);
  $db->prepare("DELETE FROM products WHERE id=?")->execute([$id]);
  flash_set('ok', 'Producto eliminado');
  redirect('/admin/products');
});

route('GET', '/admin/orders', function() use ($db) {
  admin_require();
  $orders = $db->query("SELECT * FROM orders ORDER BY created_at DESC LIMIT 200")->fetchAll();
  include __DIR__ . '/../app/views/admin/orders.php';
});

route('GET', '/admin/orders/{id}', function($params) use ($db) {
  admin_require();
  $id = (int)$params['id'];
  $st = $db->prepare("SELECT * FROM orders WHERE id=?");
  $st->execute([$id]);
  $o = $st->fetch();
  if (!$o) { http_response_code(404); echo "Pedido no encontrado"; exit; }
  $it = $db->prepare("SELECT * FROM order_items WHERE order_id=?");
  $it->execute([$id]);
  $items = $it->fetchAll();
  include __DIR__ . '/../app/views/admin/order_view.php';
});

route('POST', '/admin/orders/status', function() use ($db, $config) {
  admin_require();
  require_post();
  if (!csrf_check($config)) { http_response_code(403); echo "CSRF inválido"; exit; }
  $id = (int)($_POST['id'] ?? 0);
  $status = trim($_POST['status'] ?? 'new');
  $db->prepare("UPDATE orders SET status=? WHERE id=?")->execute([$status, $id]);
  flash_set('ok', 'Estado actualizado');
  redirect('/admin/orders/' . $id);
});

require_once __DIR__ . '/../app/services/importer.php';
require_once __DIR__ . '/../app/services/csv_importer.php';

route('GET', '/admin/import', function() use ($db, $config) {
  admin_require();
  $api_url = $config['products_api_url'] ?: db_setting_get($db, 'products_api_url', '');
  include __DIR__ . '/../app/views/admin/import.php';
});

route('POST', '/admin/import', function() use ($db, $config) {
  admin_require();
  require_post();
  if (!csrf_check($config)) { http_response_code(403); echo "CSRF inválido"; exit; }

  $api_url = trim($_POST['api_url'] ?? '');
  if ($api_url !== '') {
    db_setting_set($db, 'products_api_url', $api_url);
  } else {
    $api_url = $config['products_api_url'] ?: db_setting_get($db, 'products_api_url', '');
  }

  [$ok, $msg, $count] = import_products_from_api($db, $config, $api_url);
  if ($ok) flash_set('ok', "Import OK: $count productos");
  else flash_set('error', "Import error: $msg");
  redirect('/admin/import');
});

route('GET', '/admin/import-file', function() use ($db, $config) {
  admin_require();
  include __DIR__ . '/../app/views/admin/import_file.php';
});

route('POST', '/admin/import-file', function() use ($db, $config) {
  admin_require();
  require_post();
  if (!csrf_check($config)) { http_response_code(403); echo "CSRF inválido"; exit; }

  if (empty($_FILES['csv_file']) || $_FILES['csv_file']['error'] !== UPLOAD_ERR_OK) {
    flash_set('error', 'Error subiendo el archivo');
    redirect('/admin/import-file');
  }

  $tmp = $_FILES['csv_file']['tmp_name'];
  $name = $_FILES['csv_file']['name'] ?? 'import.csv';
  $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
  if ($ext !== 'csv') {
    flash_set('error', 'Subí un archivo .CSV (si tenés XLSX, exportalo a CSV)');
    redirect('/admin/import-file');
  }

  [$ok, $msg, $count] = import_products_from_csv($db, $tmp);
  if ($ok) flash_set('ok', "Import OK: $count productos");
  else flash_set('error', "Import error: $msg");
  redirect('/admin/import-file');
});

dispatch($_SERVER['REQUEST_METHOD'] ?? 'GET', $_SERVER['REQUEST_URI'] ?? '/');

